/*
 * Decompiled with CFR 0.152.
 */
package com.icodici.crypto;

import com.icodici.crypto.AbstractKey;
import com.icodici.crypto.EncryptionError;
import com.icodici.crypto.PBKDF2;
import com.icodici.crypto.PrivateKey;
import com.icodici.crypto.PublicKey;
import com.icodici.crypto.SymmetricKey;
import com.icodici.crypto.digest.Sha1;
import com.icodici.crypto.digest.Sha256;
import com.icodici.crypto.digest.Sha512;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import net.sergeych.boss.Boss;
import net.sergeych.utils.Base64;

public class KeyInfo {
    private byte[] salt;
    private byte[] tag = null;
    private int rounds = 0;
    private Algorythm algorythm;
    private PRF prf = PRF.None;
    private int keyLength;

    public byte[] getSalt() {
        return this.salt;
    }

    public boolean matchType(KeyInfo otherInfo) {
        if (this.keyLength != otherInfo.keyLength) {
            return false;
        }
        if (otherInfo.algorythm == this.algorythm && this.algorythm != Algorythm.RSAPublic) {
            return true;
        }
        return this.algorythm == Algorythm.RSAPrivate && otherInfo.algorythm == Algorythm.RSAPublic;
    }

    public boolean matchType(AbstractKey otherKey) {
        return this.matchType(otherKey.info());
    }

    public boolean matchTag(KeyInfo keyInfo) {
        return this.tag != null && keyInfo.tag != null && Arrays.equals(this.tag, keyInfo.tag);
    }

    public void setTag(byte[] tag) {
        this.tag = tag;
    }

    public String getBase64Tag() {
        return Base64.encodeCompactString(this.getTag());
    }

    public boolean isPassword() {
        return this.prf != PRF.None;
    }

    public KeyInfo(Algorythm algorythm, byte[] tag, int keyLength) {
        this.algorythm = algorythm;
        this.tag = tag;
        this.keyLength = keyLength;
        this.checkSanity();
    }

    public KeyInfo(Algorythm algorythm, byte[] tag) {
        this.algorythm = algorythm;
        this.tag = tag;
        this.keyLength = this.keyLength;
        if (algorythm != Algorythm.RSAPrivate && algorythm != Algorythm.RSAPublic) {
            throw new IllegalArgumentException("this algorythm requires block size");
        }
        this.checkSanity();
    }

    public KeyInfo(PRF PRF2, int rounds, byte[] salt, byte[] tag) {
        this.algorythm = Algorythm.AES256;
        this.tag = tag;
        this.prf = PRF2;
        this.rounds = rounds;
        this.salt = salt;
        this.checkSanity();
    }

    private void checkSanity() {
        switch (this.algorythm) {
            case RSAPrivate: 
            case RSAPublic: {
                if (!this.isPassword()) break;
                throw new IllegalArgumentException("RSA keys can't be password-derived");
            }
            case AES256: {
                this.keyLength = 32;
            }
        }
        if (this.isPassword()) {
            if (this.rounds < 100) {
                throw new IllegalArgumentException("should be more than 1000 rounds for PRF");
            }
            if (this.keyLength < 16) {
                throw new IllegalArgumentException("key should be at least 16 bytes for PRF");
            }
            if (this.salt == null) {
                this.salt = "attesta".getBytes();
            }
        }
    }

    public KeyInfo(byte[] packedInfo) throws IOException {
        Boss.Reader r = new Boss.Reader(packedInfo);
        this.algorythm = Algorythm.values()[r.readInt()];
        this.tag = r.readBinary();
        this.prf = PRF.values()[r.readInt()];
        this.keyLength = r.readInt();
        if (this.isPassword()) {
            if (r.readInt() != 0) {
                throw new IllegalArgumentException("unknown PBKDF type");
            }
            this.rounds = r.readInt();
        }
        this.checkSanity();
    }

    public byte[] getTag() {
        return this.tag;
    }

    public int getRounds() {
        return this.rounds;
    }

    public Algorythm getAlgorythm() {
        return this.algorythm;
    }

    public PRF getPRF() {
        return this.prf;
    }

    public int getKeyLength() {
        return this.keyLength;
    }

    public byte[] pack() {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Boss.Writer w = new Boss.Writer(baos);
        try {
            w.write(this.algorythm.ordinal(), this.tag, this.prf.ordinal(), this.keyLength);
            if (this.isPassword()) {
                w.write(0, this.rounds);
            }
            w.close();
            return baos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException("unexpected IO exception", e);
        }
    }

    public SymmetricKey derivePassword(String password) {
        Class cls;
        if (!this.isPassword()) {
            throw new IllegalStateException("not the PRF keyInfo");
        }
        switch (this.prf) {
            case HMAC_SHA1: {
                cls = Sha1.class;
                break;
            }
            case HMAC_SHA256: {
                cls = Sha256.class;
                break;
            }
            case HMAC_SHA512: {
                cls = Sha512.class;
                break;
            }
            default: {
                throw new IllegalArgumentException("unknown hash scheme for pbkdf2");
            }
        }
        byte[] key = PBKDF2.derive(cls, password, this.salt, this.rounds, this.keyLength);
        return new SymmetricKey(key, this);
    }

    public AbstractKey unpackKey(byte[] data) throws EncryptionError {
        switch (this.algorythm) {
            case RSAPublic: {
                return new PublicKey(data, this);
            }
            case RSAPrivate: {
                return new PrivateKey(data, this);
            }
            case AES256: {
                return new SymmetricKey(data, this);
            }
        }
        throw new EncryptionError("can't unpack key: " + this);
    }

    public String toString() {
        return String.format("Key(%s,%s,%s)", new Object[]{this.algorythm, this.prf, this.tag == null ? "null" : Base64.encodeCompactString(this.tag)});
    }

    public static enum PRF {
        None,
        HMAC_SHA1,
        HMAC_SHA256,
        HMAC_SHA512;

    }

    public static enum Algorythm {
        UNKNOWN,
        RSAPublic,
        RSAPrivate,
        AES256;

    }
}

