/*
 * Decompiled with CFR 0.152.
 */
package com.icodici.universa.node2.network;

import com.icodici.crypto.EncryptionError;
import com.icodici.crypto.HashType;
import com.icodici.crypto.PrivateKey;
import com.icodici.crypto.PublicKey;
import com.icodici.crypto.SymmetricKey;
import com.icodici.crypto.digest.Crc32;
import com.icodici.universa.Errors;
import com.icodici.universa.node2.NetConfig;
import com.icodici.universa.node2.NodeInfo;
import com.icodici.universa.node2.network.DatagramAdapter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.Function;
import net.sergeych.boss.Boss;
import net.sergeych.tools.Binder;
import net.sergeych.tools.Do;
import net.sergeych.utils.Bytes;
import net.sergeych.utils.LogPrinter;

public class UDPAdapter
extends DatagramAdapter {
    private static LogPrinter log = new LogPrinter("UDPA");
    private DatagramSocket socket;
    private SocketListenThread socketListenThread;
    private Object lock = new Object();
    private ConcurrentHashMap<Integer, Session> sessionsById = new ConcurrentHashMap();
    private Timer timer = new Timer();
    private Timer timerCleanup = new Timer();
    private Timer heartBeatTimer = new Timer();
    public static final int HEART_BEAT_TIME = 20000;
    public static final int CLEANUP_TIME = 15000;
    private boolean isShuttingDown = false;
    protected String label = null;

    public UDPAdapter(PrivateKey ownPrivateKey, SymmetricKey sessionKey, NodeInfo myNodeInfo, NetConfig netConfig) throws IOException {
        super(ownPrivateKey, sessionKey, myNodeInfo, netConfig);
        this.label = myNodeInfo.getNumber() + "-0: ";
        this.socket = new DatagramSocket(myNodeInfo.getNodeAddress().getPort());
        this.socket.setReuseAddress(true);
        this.socketListenThread = new SocketListenThread(this.socket);
        this.socketListenThread.start();
        this.timer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                UDPAdapter.this.checkUnsent();
            }
        }, 250L, 250L);
        this.timerCleanup.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                UDPAdapter.this.socketListenThread.cleanObtainedBlocks();
            }
        }, 15000L, 15000L);
        this.heartBeatTimer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                UDPAdapter.this.heartBeat();
            }
        }, 20000L, 20000L);
    }

    @Override
    public synchronized void send(NodeInfo destination, byte[] payload) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send to ", destination.getNumber(), ", is shutting down: ", this.isShuttingDown), 1);
        if (!this.isShuttingDown) {
            Session session = this.sessionsById.get(destination.getNumber());
            Block rawBlock = new Block(this.myNodeInfo.getNumber(), destination.getNumber(), new Random().nextInt(Integer.MAX_VALUE), -1, destination.getNodeAddress().getAddress(), destination.getNodeAddress().getPort(), (byte[])payload.clone());
            if (session != null) {
                if (session.isValid().booleanValue()) {
                    if (session.state == 2 || session.state == 6) {
                        this.report(this.getLabel(), "session is ok", 1);
                        session.addBlockToWaitingQueue(rawBlock);
                        this.sendAsDataBlock(rawBlock, session);
                    } else {
                        this.report(this.getLabel(), "session is handshaking", 1);
                        session.addBlockToWaitingQueue(rawBlock);
                    }
                } else if (session.state == 2 || session.state == 6) {
                    this.report(this.getLabel(), "session not valid for exchanging, recreate", 1);
                    if (this.sessionsById.containsKey(session.remoteNodeId)) {
                        this.sessionsById.remove(session.remoteNodeId);
                    }
                    session = this.getOrCreateSession(destination.getNumber(), destination.getNodeAddress().getAddress(), destination.getNodeAddress().getPort());
                    session.publicKey = destination.getPublicKey();
                    session.remoteNodeId = destination.getNumber();
                    session.addBlockToWaitingQueue(rawBlock);
                    this.sendHello(session);
                } else {
                    this.report(this.getLabel(), "session not valid yet, but it is handshaking", 1);
                    session.addBlockToWaitingQueue(rawBlock);
                }
            } else {
                this.report(this.getLabel(), "session not exist", 1);
                session = this.getOrCreateSession(destination.getNumber(), destination.getNodeAddress().getAddress(), destination.getNodeAddress().getPort());
                session.publicKey = destination.getPublicKey();
                session.remoteNodeId = destination.getNumber();
                session.addBlockToWaitingQueue(rawBlock);
                this.sendHello(session);
            }
        }
    }

    @Override
    public void shutdown() {
        this.isShuttingDown = true;
        this.report(this.getLabel(), "shutting down...", 1);
        this.socketListenThread.shutdownThread();
        this.socket.close();
        this.socket.disconnect();
        this.closeSessions();
        this.timer.cancel();
        this.timer.purge();
        this.heartBeatTimer.cancel();
        this.heartBeatTimer.purge();
        this.timerCleanup.cancel();
        this.timerCleanup.purge();
        try {
            while (this.socket.isConnected()) {
                this.report(this.getLabel(), () -> this.concatReportMessage("shutting down... ", this.socket.isClosed(), " ", this.socket.isConnected()), 1);
                Thread.sleep(100L);
            }
        }
        catch (InterruptedException e) {
            e.printStackTrace();
        }
        this.report(this.getLabel(), "shutdown", 1);
    }

    public void closeSessions() {
        this.report(this.getLabel(), "closeSessions");
        this.sessionsById.clear();
    }

    public void brakeSessions() {
        this.report(this.getLabel(), "brakeSessions");
        for (Session s : this.sessionsById.values()) {
            s.publicKey = new PublicKey();
        }
    }

    public String getLabel() {
        return this.label;
    }

    public DatagramSocket getSocket() {
        return this.socket;
    }

    public void report(String label, String message, int level) {
        if (level <= this.verboseLevel) {
            System.out.println(label + message);
        }
    }

    public void report(String label, Callable<String> message, int level) {
        if (level <= this.verboseLevel) {
            try {
                System.out.println(label + message.call());
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void report(String label, String message) {
        this.report(label, message, 2);
    }

    public void report(String label, Callable<String> message) {
        this.report(label, message, 2);
    }

    protected void sendBlock(Block block, Session session) throws InterruptedException {
        if (!block.isValidToSend().booleanValue()) {
            block.prepareToSend(512);
        }
        ArrayList outs = new ArrayList(block.datagrams.values());
        block.sendAttempts++;
        if (block.type != 7 && block.type != 1 && block.type != 2) {
            session.addBlockToSendingQueue(block);
        }
        try {
            if (this.testMode == 2 || this.testMode == 3) {
                Collections.shuffle(outs);
            }
            this.report(this.getLabel(), () -> this.concatReportMessage("for block: ", block.blockId, " sending packets num:  ", outs.size()));
            for (DatagramPacket d : outs) {
                if ((this.testMode == 1 || this.testMode == 3) && new Random().nextInt(100) < this.lostPacketsPercent) {
                    this.report(this.getLabel(), () -> this.concatReportMessage("Lost packet in block: ", block.blockId));
                    continue;
                }
                this.socket.send(d);
                this.report(this.getLabel(), () -> this.concatReportMessage("for block: ", block.blockId, " sent packets num:  ", outs.size()));
            }
        }
        catch (IOException e) {
            this.report(this.getLabel(), "send block error, socket already closed");
        }
    }

    protected synchronized void sendAsDataBlock(Block rawDataBlock, Session session) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send data to ", session.remoteNodeId), 1);
        this.report(this.getLabel(), () -> this.concatReportMessage("sessionKey is ", session.sessionKey.hashCode(), " for ", session.remoteNodeId));
        byte[] crc32Local = new Crc32().digest(rawDataBlock.payload);
        this.report(this.getLabel(), () -> this.concatReportMessage("sendAsDataBlock: Crc32 id is ", Arrays.equals(rawDataBlock.crc32, crc32Local)));
        try {
            byte[] encrypted = session.sessionKey.etaEncrypt((byte[])rawDataBlock.payload.clone());
            Binder binder = Binder.fromKeysValues("data", encrypted, "crc32", rawDataBlock.crc32);
            byte[] packedData = Boss.pack(binder);
            this.report(this.getLabel(), () -> this.concatReportMessage(" data size: ", rawDataBlock.payload.length, " for ", session.remoteNodeId));
            Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, rawDataBlock.blockId, 0, session.address, session.port, packedData);
            this.sendBlock(block, session);
        }
        catch (EncryptionError encryptionError) {
            this.callErrorCallbacks("[sendAsDataBlock] EncryptionError in node " + this.myNodeInfo.getNumber() + ": " + encryptionError.getMessage());
            if (this.sessionsById.containsKey(session.remoteNodeId)) {
                this.sessionsById.remove(session.remoteNodeId);
            }
            Session newSession = this.getOrCreateSession(session.remoteNodeId, session.address, session.port);
            newSession.publicKey = session.publicKey;
            newSession.remoteNodeId = session.remoteNodeId;
            newSession.addBlockToWaitingQueue(rawDataBlock);
            this.sendHello(newSession);
        }
    }

    protected void sendHello(Session session) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send hello to ", session.remoteNodeId), 1);
        session.state = 3;
        Binder binder = Binder.fromKeysValues("data", this.myNodeInfo.getNumber());
        Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 3, session.address, session.port, Boss.pack(binder));
        this.sendBlock(block, session);
    }

    protected void sendWelcome(Session session) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send welcome to ", session.remoteNodeId), 1);
        session.state = 4;
        Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 4, session.address, session.port, session.localNonce);
        this.sendBlock(block, session);
    }

    protected synchronized void sendKeyRequest(Session session) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send key request to ", session.remoteNodeId), 1);
        session.state = 5;
        List data = Arrays.asList(session.localNonce, session.remoteNonce);
        try {
            byte[] packed = Boss.pack(data);
            byte[] signed = this.ownPrivateKey.sign(packed, HashType.SHA512);
            Binder binder = Binder.fromKeysValues("data", packed, "signature", signed);
            Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 5, session.address, session.port, Boss.pack(binder));
            this.sendBlock(block, session);
        }
        catch (EncryptionError encryptionError) {
            this.callErrorCallbacks("[sendKeyRequest] EncryptionError in node " + this.myNodeInfo.getNumber() + ": " + encryptionError.getMessage());
            if (this.sessionsById.containsKey(session.remoteNodeId)) {
                this.sessionsById.remove(session.remoteNodeId);
            }
            Session newSession = this.getOrCreateSession(session.remoteNodeId, session.address, session.port);
            newSession.publicKey = session.publicKey;
            newSession.remoteNodeId = session.remoteNodeId;
            for (Block b : session.waitingBlocksQueue) {
                newSession.addBlockToWaitingQueue(b);
            }
            this.sendHello(newSession);
        }
    }

    protected synchronized void sendSessionKey(Session session) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send session key to ", session.remoteNodeId), 1);
        this.report(this.getLabel(), () -> this.concatReportMessage("sessionKey is ", session.sessionKey.hashCode(), " for ", session.remoteNodeId));
        List data = Arrays.asList(session.sessionKey.getKey(), session.remoteNonce);
        try {
            byte[] packed = Boss.pack(data);
            PublicKey sessionPublicKey = new PublicKey(session.publicKey.pack());
            byte[] encrypted = sessionPublicKey.encrypt(packed);
            byte[] signed = this.ownPrivateKey.sign(encrypted, HashType.SHA512);
            Binder binder = Binder.fromKeysValues("data", encrypted, "signature", signed);
            Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 6, session.address, session.port, Boss.pack(binder));
            this.sendBlock(block, session);
            session.state = 6;
        }
        catch (EncryptionError encryptionError) {
            this.callErrorCallbacks("[sendSessionKey] EncryptionError in node " + this.myNodeInfo.getNumber() + ": " + encryptionError.getMessage());
            if (this.sessionsById.containsKey(session.remoteNodeId)) {
                this.sessionsById.remove(session.remoteNodeId);
            }
            Session newSession = this.getOrCreateSession(session.remoteNodeId, session.address, session.port);
            newSession.publicKey = session.publicKey;
            newSession.remoteNodeId = session.remoteNodeId;
            for (Block b : session.waitingBlocksQueue) {
                newSession.addBlockToWaitingQueue(b);
            }
            this.sendHello(newSession);
        }
    }

    protected void sendPacketAck(Session session, int blockId, int packetId) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send packet_ack to ", session.remoteNodeId));
        List<Integer> data = Arrays.asList(blockId, packetId);
        Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 7, session.address, session.port, Boss.pack(data));
        this.sendBlock(block, session);
    }

    protected void sendAck(Session session, int blockId) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send ack to ", session.remoteNodeId), 1);
        Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 1, session.address, session.port, Boss.pack(blockId));
        this.sendBlock(block, session);
    }

    protected void sendNack(Session session, int blockId) throws InterruptedException {
        this.report(this.getLabel(), () -> this.concatReportMessage("send nack to ", session.remoteNodeId), 1);
        Block block = new Block(this.myNodeInfo.getNumber(), session.remoteNodeId, new Random().nextInt(Integer.MAX_VALUE), 2, session.address, session.port, Boss.pack(blockId));
        this.sendBlock(block, session);
    }

    protected Session getOrCreateSession(int remoteId, InetAddress address, int port) {
        if (this.sessionsById.containsKey(remoteId)) {
            Session s = this.sessionsById.get(remoteId);
            this.report(this.getLabel(), () -> this.concatReportMessage(">>Session was exist for node ", remoteId, " at the node ", this.myNodeInfo.getNumber()), 1);
            this.report(this.getLabel(), () -> this.concatReportMessage(">>local node: ", this.myNodeInfo.getNumber(), " remote node: ", s.remoteNodeId), 1);
            this.report(this.getLabel(), () -> this.concatReportMessage(">>local nonce: ", s.localNonce, " remote nonce: ", s.remoteNonce), 1);
            this.report(this.getLabel(), () -> this.concatReportMessage(">>state: ", s.state), 1);
            this.report(this.getLabel(), () -> this.concatReportMessage(">>session key: ", s.sessionKey.hashCode()), 1);
            return s;
        }
        Session session = new Session(address, port);
        this.report(this.getLabel(), () -> this.concatReportMessage("session created for nodeId ", remoteId), 1);
        session.remoteNodeId = remoteId;
        session.sessionKey = this.sessionKey;
        this.report(this.getLabel(), () -> this.concatReportMessage("sessionKey is ", session.sessionKey.hashCode(), " localNonce is ", session.localNonce, " for ", session.remoteNodeId), 1);
        this.sessionsById.putIfAbsent(remoteId, session);
        return session;
    }

    protected void checkUnsent() {
        ArrayList<Session> brokenSessions = new ArrayList<Session>();
        for (Session session : this.sessionsById.values()) {
            ArrayList<Block> blocksToRemove = new ArrayList<Block>();
            for (Block block : session.sendingBlocksQueue) {
                if (block.isDelivered().booleanValue()) continue;
                this.report(this.getLabel(), () -> this.concatReportMessage("block: ", block.blockId, " type: ", block.type, " sendAttempts: ", block.sendAttempts, " not delivered"));
                try {
                    if (block.sendAttempts >= 50) {
                        this.report(this.getLabel(), () -> this.concatReportMessage("block ", block.blockId, " type ", block.type, " will be removed"));
                        blocksToRemove.add(block);
                        continue;
                    }
                    this.sendBlock(block, session);
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            for (Block rb : blocksToRemove) {
                try {
                    session.removeBlockFromSendingQueue(rb);
                    session.removeBlockFromWaitingQueue(rb.blockId);
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            if (blocksToRemove.size() <= 0) continue;
            this.report(this.getLabel(), () -> this.concatReportMessage("Session with remote ", session.remoteNodeId, " is possible broken, state: ", session.state, ", num sending: ", session.sendingBlocksQueue.size(), ", num waiting: ", session.waitingBlocksQueue.size()));
            if (session.state == 2 || session.state == 6 || !session.sendingBlocksQueue.isEmpty()) continue;
            this.report(this.getLabel(), () -> this.concatReportMessage("Session with remote ", session.remoteNodeId, " is broken, state: ", session.state, ", num sending: ", session.sendingBlocksQueue.size(), ", num waiting: ", session.waitingBlocksQueue.size(), ", will be removed"));
            brokenSessions.add(session);
        }
        for (Session session : brokenSessions) {
            if (!this.sessionsById.containsKey(session.remoteNodeId)) continue;
            this.sessionsById.remove(session.remoteNodeId);
        }
    }

    protected void checkUnsentPackets(Session session) {
        ArrayList<Block> blocksToResend = new ArrayList<Block>();
        ArrayList packetsToResend = new ArrayList();
        ArrayList datagramsToResend = new ArrayList();
        for (Block block : session.sendingBlocksQueue) {
            if (block.isDelivered().booleanValue()) continue;
            blocksToResend.add(block);
            for (Packet packet : block.packets.values()) {
                if (packet.sendWaitIndex < 3) continue;
                datagramsToResend.add(block.datagrams.get(packet.packetId));
                this.report(this.getLabel(), () -> this.concatReportMessage("packet will be resend, blockId: ", packet.blockId, " packetId: ", packet.packetId, " type: ", packet.type, " sendWaitIndex: ", packet.sendWaitIndex));
            }
        }
        for (DatagramPacket datagram : datagramsToResend) {
            try {
                if (datagram != null) {
                    this.socket.send(datagram);
                    this.report(this.getLabel(), " datagram was resent");
                    continue;
                }
                this.report(this.getLabel(), " datagram unexpected became null");
            }
            catch (IOException iOException) {}
        }
    }

    protected void sendWaitingBlocks(Session session) {
        if (session != null && session.isValid().booleanValue() && (session.state == 2 || session.state == 6)) {
            this.report(this.getLabel(), () -> this.concatReportMessage("waiting blocks num ", session.waitingBlocksQueue.size()));
            try {
                for (Block waitingBlock : session.waitingBlocksQueue) {
                    this.report(this.getLabel(), () -> this.concatReportMessage("waitingBlock ", waitingBlock.blockId, " type ", waitingBlock.type));
                    if (waitingBlock.type == -1) {
                        this.sendAsDataBlock(waitingBlock, session);
                        continue;
                    }
                    this.sendBlock(waitingBlock, session);
                }
            }
            catch (InterruptedException e) {
                System.out.println((Object)((Object)Errors.FAILURE) + " sending waiting blocks interrupted, " + e.getMessage());
            }
        }
    }

    protected void callErrorCallbacks(String message) {
        for (Function fn : this.errorCallbacks) {
            fn.apply(message);
        }
    }

    protected void heartBeat() {
        int waitingBlocksNum = 0;
        int sendingBlocksNum = 0;
        int sendingPacketsNum = 0;
        for (Session session : this.sessionsById.values()) {
            waitingBlocksNum += session.waitingBlocksQueue.size();
            sendingBlocksNum += session.sendingBlocksQueue.size();
            sendingPacketsNum += session.sendingPacketsQueue.size();
        }
        boolean finalwaitingBlocksNum = false;
        boolean finalsendingBlocksNum = false;
        boolean finalsendingPacketsNum = false;
        this.report(this.getLabel(), () -> this.concatReportMessage("heartbeat: [sessions: ", this.sessionsById.size(), ", waiting blocks: ", 0, ", sending blocks: ", 0, ", sending packets: ", 0, "]"), 1);
    }

    protected String concatReportMessage(Object ... messages) {
        String returnMessage = "";
        for (Object m : messages) {
            returnMessage = returnMessage + (m != null ? m.toString() : "null");
        }
        return returnMessage;
    }

    public Block createTestBlock(int senderNodeId, int receiverNodeId, int blockId, int type, InetAddress address, int port, byte[] payload) {
        return new Block(senderNodeId, receiverNodeId, blockId, type, address, port, payload);
    }

    private class Session {
        private PublicKey publicKey;
        private SymmetricKey sessionKey;
        private InetAddress address;
        private int port;
        private byte[] localNonce;
        private byte[] remoteNonce;
        private int remoteNodeId = -1;
        private int state;
        public static final int NOT_EXIST = 0;
        public static final int HANDSHAKE = 1;
        public static final int EXCHANGING = 2;
        public static final int HELLO = 3;
        public static final int WELCOME = 4;
        public static final int KEY_REQ = 5;
        public static final int SESSION = 6;
        private BlockingQueue<Block> waitingBlocksQueue = new LinkedBlockingQueue<Block>();
        private BlockingQueue<Block> sendingBlocksQueue = new LinkedBlockingQueue<Block>();
        private BlockingQueue<Packet> sendingPacketsQueue = new LinkedBlockingQueue<Packet>();

        Session(InetAddress address, int port) {
            this.address = address;
            this.port = port;
            this.localNonce = Do.randomBytes(64);
            this.state = 1;
        }

        public Boolean isValid() {
            if (this.localNonce == null) {
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), "session validness check: localNonce is null");
                return false;
            }
            if (this.remoteNonce == null) {
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), "session validness check: remoteNonce is null");
                return false;
            }
            if (this.sessionKey == null) {
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), "session validness check: sessionKey is null");
                return false;
            }
            if (this.remoteNodeId < 0) {
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), "session validness check: remoteNodeId is not defined");
                return false;
            }
            return true;
        }

        public void createSessionKey() throws EncryptionError {
            if (this.sessionKey == null) {
                this.sessionKey = new SymmetricKey();
            }
        }

        public void reconstructSessionKey(byte[] key) throws EncryptionError {
            this.sessionKey = new SymmetricKey(key);
            this.state = 2;
        }

        public void addBlockToWaitingQueue(Block block) throws InterruptedException {
            if (!this.waitingBlocksQueue.contains(block)) {
                this.waitingBlocksQueue.put(block);
            }
        }

        public void addBlockToSendingQueue(Block block) throws InterruptedException {
            if (!this.sendingBlocksQueue.contains(block)) {
                this.sendingBlocksQueue.put(block);
            }
            for (Packet p : block.packets.values()) {
                this.addPacketToSendingQueue(p);
            }
        }

        public void addPacketToSendingQueue(Packet packet) throws InterruptedException {
            if (!this.sendingPacketsQueue.contains(packet)) {
                this.sendingPacketsQueue.put(packet);
            }
        }

        public void removeBlockFromWaitingQueue(Block block) throws InterruptedException {
            if (this.waitingBlocksQueue.contains(block)) {
                this.waitingBlocksQueue.remove(block);
            }
        }

        public void removeBlockFromWaitingQueue(int blockId) throws InterruptedException {
            for (Block block : this.waitingBlocksQueue) {
                if (block.blockId != blockId) continue;
                this.removeBlockFromWaitingQueue(block);
            }
        }

        public void removeBlockFromSendingQueue(Block block) throws InterruptedException {
            if (this.sendingBlocksQueue.contains(block)) {
                this.sendingBlocksQueue.remove(block);
            }
            for (Packet p : block.packets.values()) {
                this.removePacketFromSendingQueue(p);
            }
        }

        public void removePacketFromSendingQueue(Packet packet) throws InterruptedException {
            if (this.sendingPacketsQueue.contains(packet)) {
                this.sendingPacketsQueue.remove(packet);
            }
            for (Block sendingBlock : this.sendingBlocksQueue) {
                if (sendingBlock.blockId != packet.blockId) continue;
                sendingBlock.markPacketAsDelivered(packet);
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), () -> UDPAdapter.this.concatReportMessage("markPacketAsDelivered, packets num: ", sendingBlock.packets.size(), " diagrams num: ", sendingBlock.datagrams.size()));
            }
            UDPAdapter.this.report(UDPAdapter.this.getLabel(), () -> UDPAdapter.this.concatReportMessage("remove packet from queue, blockId: ", packet.blockId, " packetId: ", packet.packetId, " type: ", packet.type, " sendWaitIndex: ", packet.sendWaitIndex, " delivered: ", packet.delivered));
        }

        public void incremetWaitIndexForPacketsFromSendingQueue() throws InterruptedException {
            for (Packet p : this.sendingPacketsQueue) {
                p.sendWaitIndex++;
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), () -> UDPAdapter.this.concatReportMessage("packet, blockId: ", p.blockId, " packetId: ", p.packetId, " type: ", p.type, " sendWaitIndex: ", p.sendWaitIndex));
            }
        }

        public void removePacketFromSendingQueue(int blockId, int packetId) throws InterruptedException {
            for (Block sendingBlock : this.sendingBlocksQueue) {
                if (sendingBlock.blockId != blockId) continue;
                for (Packet p : sendingBlock.packets.values()) {
                    if (p.packetId != packetId) continue;
                    this.removePacketFromSendingQueue(p);
                }
            }
        }

        public void makeBlockDelivered(int blockId) throws InterruptedException {
            for (Block sendingBlock : this.sendingBlocksQueue) {
                if (sendingBlock.blockId != blockId) continue;
                this.removeBlockFromSendingQueue(sendingBlock);
                sendingBlock.delivered = true;
                UDPAdapter.this.report(UDPAdapter.this.getLabel(), "block " + sendingBlock.blockId + " delivered");
            }
        }

        public void moveBlocksFromSendingToWaiting() throws InterruptedException {
            for (Block sendingBlock : this.sendingBlocksQueue) {
                sendingBlock.validToSend = false;
                this.removeBlockFromSendingQueue(sendingBlock);
                this.addBlockToWaitingQueue(sendingBlock);
            }
        }

        public void removeDataBlocksFromWaiting() throws InterruptedException {
            for (Block block : this.waitingBlocksQueue) {
                if (block.type != 0) continue;
                this.removeBlockFromWaitingQueue(block);
            }
        }

        public void makeBlockDeliveredByType(int type) throws InterruptedException {
            for (Block sendingBlock : this.sendingBlocksQueue) {
                if (sendingBlock.type != type) continue;
                this.removeBlockFromSendingQueue(sendingBlock);
                sendingBlock.delivered = true;
            }
        }

        public Block getRawDataBlockFromWaitingQueue(int blockId) throws InterruptedException {
            for (Block block : this.waitingBlocksQueue) {
                if (block.blockId != blockId || block.type != -1) continue;
                return block;
            }
            return null;
        }

        static /* synthetic */ byte[] access$1402(Session x0, byte[] x1) {
            x0.remoteNonce = x1;
            return x1;
        }
    }

    public class Block {
        private int senderNodeId;
        private int receiverNodeId;
        private int blockId;
        private int type;
        private byte[] payload;
        private byte[] crc32;
        private int sendAttempts;
        private InetAddress address;
        private int port;
        private ConcurrentHashMap<Integer, Packet> packets;
        private ConcurrentHashMap<Integer, DatagramPacket> datagrams;
        private Boolean delivered = false;
        private Boolean validToSend = false;

        public Block() {
            this.packets = new ConcurrentHashMap();
            this.datagrams = new ConcurrentHashMap();
        }

        public Block(int senderNodeId, int receiverNodeId, int blockId, int type, InetAddress address, int port, byte[] payload) {
            this.senderNodeId = senderNodeId;
            this.receiverNodeId = receiverNodeId;
            this.blockId = blockId;
            this.type = type;
            this.address = address;
            this.port = port;
            this.payload = payload;
            this.crc32 = new Crc32().digest(payload);
            this.packets = new ConcurrentHashMap();
            this.datagrams = new ConcurrentHashMap();
        }

        public Block(int senderNodeId, int receiverNodeId, int blockId, int type, InetAddress address, int port) {
            this.senderNodeId = senderNodeId;
            this.receiverNodeId = receiverNodeId;
            this.blockId = blockId;
            this.type = type;
            this.address = address;
            this.port = port;
            this.packets = new ConcurrentHashMap();
            this.datagrams = new ConcurrentHashMap();
        }

        public void prepareToSend(int packetSize) {
            this.prepareToSend(packetSize, 5);
        }

        public void prepareToSend(int packetSize, int bossArtefact) {
            this.packets = new ConcurrentHashMap();
            this.datagrams = new ConcurrentHashMap();
            List<Integer> headerData = Arrays.asList(0, 0, this.senderNodeId, this.receiverNodeId, this.blockId, this.type);
            int headerSize = Boss.dump(headerData, new Object[0]).size() + bossArtefact;
            int offset = 0;
            int copySize = 0;
            int packetId = 0;
            int packetsNum = this.payload.length / (packetSize - headerSize) + 1;
            while (this.payload.length > offset) {
                copySize = packetSize - headerSize;
                if (offset + copySize >= this.payload.length) {
                    copySize = this.payload.length - offset;
                }
                byte[] cutPayload = Arrays.copyOfRange(this.payload, offset, offset + copySize);
                Packet packet = new Packet(packetsNum, packetId, this.senderNodeId, this.receiverNodeId, this.blockId, this.type, cutPayload);
                this.packets.put(packetId, packet);
                byte[] blockByteArray = packet.makeByteArray();
                if (blockByteArray.length > packetSize) {
                    this.datagrams.clear();
                    this.packets.clear();
                    this.prepareToSend(packetSize, bossArtefact + 1);
                    return;
                }
                DatagramPacket datagramPacket = new DatagramPacket(blockByteArray, blockByteArray.length, this.address, this.port);
                this.datagrams.put(packetId, datagramPacket);
                offset += copySize;
                ++packetId;
            }
            this.validToSend = true;
        }

        public void reconstruct() throws IOException {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            for (Packet packet : this.packets.values()) {
                outputStream.write(packet.payload);
            }
            this.payload = outputStream.toByteArray();
        }

        public void addToPackets(Packet packet) {
            if (!this.packets.containsKey(packet.packetId)) {
                this.packets.put(packet.packetId, packet);
            }
        }

        public void markPacketAsDelivered(Packet packet) throws InterruptedException {
            packet.delivered = true;
            if (this.datagrams.containsKey(packet.packetId)) {
                this.datagrams.remove(packet.packetId);
            }
        }

        public Boolean isSolid() {
            if (this.packets.size() > 0 && this.packets.get(0) != null && this.packets.get(0).brotherPacketsNum == this.packets.size()) {
                return true;
            }
            return false;
        }

        public Boolean isValidToSend() {
            if (this.packets.size() == 0) {
                return false;
            }
            return this.validToSend;
        }

        public Boolean isDelivered() {
            return this.delivered;
        }

        public ConcurrentHashMap<Integer, DatagramPacket> getDatagrams() {
            return this.datagrams;
        }
    }

    public class Packet {
        private int senderNodeId;
        private int receiverNodeId;
        private int blockId;
        private int packetId = 0;
        private int brotherPacketsNum = 0;
        private int type;
        private byte[] payload;
        private int sendWaitIndex = 0;
        private Boolean delivered = false;

        public Packet() {
        }

        public Packet(int packetsNum, int packetId, int senderNodeId, int receiverNodeId, int blockId, int type, byte[] payload) {
            this.brotherPacketsNum = packetsNum;
            this.packetId = packetId;
            this.senderNodeId = senderNodeId;
            this.receiverNodeId = receiverNodeId;
            this.blockId = blockId;
            this.type = type;
            this.payload = payload;
        }

        public byte[] makeByteArray() {
            List<Serializable> data = Arrays.asList(this.brotherPacketsNum, this.packetId, this.senderNodeId, this.receiverNodeId, this.blockId, this.type, new Bytes(new byte[][]{this.payload}));
            Bytes byteArray = Boss.dump(data, new Object[0]);
            return byteArray.toArray();
        }

        public void parseFromByteArray(byte[] byteArray) throws IOException {
            List data = (List)Boss.load(byteArray);
            this.brotherPacketsNum = (Integer)data.get(0);
            this.packetId = (Integer)data.get(1);
            this.senderNodeId = (Integer)data.get(2);
            this.receiverNodeId = (Integer)data.get(3);
            this.blockId = (Integer)data.get(4);
            this.type = (Integer)data.get(5);
            this.payload = ((Bytes)data.get(6)).toArray();
        }
    }

    public class PacketTypes {
        public static final int RAW_DATA = -1;
        public static final int DATA = 0;
        public static final int ACK = 1;
        public static final int NACK = 2;
        public static final int HELLO = 3;
        public static final int WELCOME = 4;
        public static final int KEY_REQ = 5;
        public static final int SESSION = 6;
        public static final int PACKET_ACK = 7;
    }

    class SocketListenThread
    extends Thread {
        private Boolean active = false;
        private final DatagramSocket threadSocket;
        private DatagramPacket receivedDatagram;
        private ConcurrentHashMap<Integer, Block> waitingBlocks = new ConcurrentHashMap();
        private ConcurrentHashMap<Integer, Instant> obtainedBlocks = new ConcurrentHashMap();
        private ConcurrentLinkedQueue<BlockTime> obtainedBlocksQueue = new ConcurrentLinkedQueue();
        private Duration maxObtainedBlockAge = Duration.ofMinutes(5L);
        protected String label = null;

        public SocketListenThread(DatagramSocket socket) {
            byte[] buf = new byte[512];
            this.receivedDatagram = new DatagramPacket(buf, buf.length);
            this.threadSocket = socket;
        }

        public void cleanObtainedBlocks() {
            Instant now = Instant.now();
            BlockTime blockTime = this.obtainedBlocksQueue.peek();
            while (blockTime != null && blockTime.expiresAt.isBefore(now)) {
                this.obtainedBlocks.remove(blockTime.blockId);
                this.obtainedBlocksQueue.poll();
                blockTime = this.obtainedBlocksQueue.peek();
            }
        }

        @Override
        public void run() {
            this.setName("UDP-socket-listener-" + Integer.toString(new Random().nextInt(100)));
            this.label = UDPAdapter.this.myNodeInfo.getNumber() + "-" + this.getName() + ": ";
            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" UDPAdapter listen socket at ", UDPAdapter.this.myNodeInfo.getNodeAddress().getAddress(), ":", UDPAdapter.this.myNodeInfo.getNodeAddress().getPort()));
            this.active = true;
            while (this.active.booleanValue()) {
                try {
                    if (!this.threadSocket.isClosed() && this.active.booleanValue()) {
                        this.threadSocket.receive(this.receivedDatagram);
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(">>>> got data"));
                    }
                }
                catch (SocketException socketException) {
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
                if (this.active.booleanValue()) {
                    byte[] data = Arrays.copyOfRange(this.receivedDatagram.getData(), 0, this.receivedDatagram.getLength());
                    Packet packet = new Packet();
                    Block waitingBlock = null;
                    try {
                        packet.parseFromByteArray(data);
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got packet with blockId: ", packet.blockId, " packetId: ", packet.packetId, " type: ", packet.type));
                        if (this.waitingBlocks.containsKey(packet.blockId)) {
                            waitingBlock = this.waitingBlocks.get(packet.blockId);
                        } else if (this.obtainedBlocks.containsKey(packet.blockId)) {
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" warning: repeated block given, with id ", packet.blockId));
                        } else {
                            waitingBlock = new Block(packet.senderNodeId, packet.receiverNodeId, packet.blockId, packet.type, this.receivedDatagram.getAddress(), this.receivedDatagram.getPort());
                            this.waitingBlocks.put(waitingBlock.blockId, waitingBlock);
                        }
                        if (waitingBlock == null) continue;
                        waitingBlock.addToPackets(packet);
                        if (waitingBlock.isSolid().booleanValue()) {
                            this.moveWaitingBlockToObtained(waitingBlock);
                            waitingBlock.reconstruct();
                            this.obtainSolidBlock(waitingBlock);
                            continue;
                        }
                        if (packet.type == 7) continue;
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got packet type: ", packet.type, " brotherPacketsNum: ", packet.brotherPacketsNum, " from ", packet.senderNodeId), 1);
                        Session session = UDPAdapter.this.getOrCreateSession(packet.senderNodeId, this.receivedDatagram.getAddress(), this.receivedDatagram.getPort());
                        UDPAdapter.this.sendPacketAck(session, packet.blockId, packet.packetId);
                        switch (packet.type) {
                            case 3: {
                                session.makeBlockDeliveredByType(3);
                                break;
                            }
                            case 4: {
                                session.makeBlockDeliveredByType(3);
                                session.makeBlockDeliveredByType(4);
                                break;
                            }
                            case 5: {
                                session.makeBlockDeliveredByType(3);
                                session.makeBlockDeliveredByType(4);
                                session.makeBlockDeliveredByType(5);
                                break;
                            }
                            case 6: {
                                session.makeBlockDeliveredByType(3);
                                session.makeBlockDeliveredByType(4);
                                session.makeBlockDeliveredByType(5);
                                session.makeBlockDeliveredByType(6);
                            }
                            case 0: {
                                if (!session.isValid().booleanValue() || session.state != 2 && session.state != 6) break;
                                session.makeBlockDeliveredByType(3);
                                session.makeBlockDeliveredByType(4);
                                session.makeBlockDeliveredByType(5);
                                session.makeBlockDeliveredByType(6);
                            }
                        }
                        continue;
                    }
                    catch (IllegalArgumentException e) {
                        e.printStackTrace();
                        continue;
                    }
                    catch (InterruptedException e) {
                        UDPAdapter.this.report(this.getLabel(), "expected interrupted exception");
                        continue;
                    }
                    catch (SymmetricKey.AuthenticationFailed e) {
                        UDPAdapter.this.callErrorCallbacks("SymmetricKey.AuthenticationFailed in node " + UDPAdapter.this.myNodeInfo.getNumber() + ": " + e.getMessage());
                        e.printStackTrace();
                        continue;
                    }
                    catch (EncryptionError e) {
                        UDPAdapter.this.callErrorCallbacks(this.getLabel() + " EncryptionError in node " + UDPAdapter.this.myNodeInfo.getNumber() + ": " + e.getMessage());
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("EncryptionError in node ", UDPAdapter.this.myNodeInfo.getNumber(), ": ", e.getMessage()), 1);
                        for (Session s : UDPAdapter.this.sessionsById.values()) {
                            UDPAdapter.this.report(this.getLabel(), ">>---", 1);
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(">>local node: ", UDPAdapter.this.myNodeInfo.getNumber(), " remote node: ", s.remoteNodeId), 1);
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(">>local nonce: ", s.localNonce, " remote nonce: ", s.remoteNonce), 1);
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(">>state: ", s.state), 1);
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(">>session key: ", s.sessionKey.hashCode()), 1);
                        }
                        continue;
                    }
                    catch (IOException e) {
                        e.printStackTrace();
                        continue;
                    }
                    catch (IllegalStateException e) {
                        UDPAdapter.this.callErrorCallbacks("IllegalStateException in node " + UDPAdapter.this.myNodeInfo.getNumber() + ": " + e.getMessage());
                        e.printStackTrace();
                        continue;
                    }
                }
                UDPAdapter.this.report(this.getLabel(), "socket will be closed");
                this.shutdownThread();
            }
        }

        public void shutdownThread() {
            this.active = false;
            this.interrupt();
            this.threadSocket.close();
        }

        public String getLabel() {
            return this.label;
        }

        protected void obtainSolidBlock(Block block) throws SymmetricKey.AuthenticationFailed, EncryptionError, InterruptedException {
            Session session = null;
            switch (block.type) {
                case 3: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got hello from ", block.senderNodeId), 1);
                    NodeInfo senderNodeInfo = UDPAdapter.this.netConfig.getInfo(block.senderNodeId);
                    if (senderNodeInfo != null) {
                        PublicKey key = senderNodeInfo.getPublicKey();
                        session = UDPAdapter.this.getOrCreateSession(block.senderNodeId, block.address, block.port);
                        session.publicKey = key;
                        session.makeBlockDeliveredByType(3);
                        if (session.state == 1 || session.state == 2 || session.state == 6) {
                            UDPAdapter.this.sendWelcome(session);
                            break;
                        }
                        int sessionRemoteNodeId = session.remoteNodeId;
                        int sessionState = session.state;
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("node sent handshake too, to ", sessionRemoteNodeId, " state: ", sessionState), 1);
                        if (session.state == 3) {
                            if (UDPAdapter.this.myNodeInfo.getNumber() >= session.remoteNodeId) break;
                            UDPAdapter.this.sendWelcome(session);
                            break;
                        }
                        this.downStateAndResend(session);
                        break;
                    }
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("Block from unknown node ", block.senderNodeId, " was already obtained, will remove from obtained"));
                    if (this.obtainedBlocks.containsKey(block.blockId)) {
                        this.obtainedBlocks.remove(block.blockId);
                    }
                    throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": block got from unknown node " + block.senderNodeId);
                }
                case 4: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got welcome from ", block.senderNodeId), 1);
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    if (session != null) {
                        Session.access$1402(session, block.payload);
                        session.makeBlockDeliveredByType(3);
                        session.makeBlockDeliveredByType(4);
                        if (session.state == 3) {
                            UDPAdapter.this.sendKeyRequest(session);
                            break;
                        }
                        int sessionRemoteNodeId = session.remoteNodeId;
                        int sessionState = session.state;
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("node sent handshake too, to ", sessionRemoteNodeId, " state: ", sessionState), 1);
                        if (session.state == 4) {
                            if (UDPAdapter.this.myNodeInfo.getNumber() >= session.remoteNodeId) break;
                            UDPAdapter.this.sendKeyRequest(session);
                            break;
                        }
                        this.downStateAndResend(session);
                        break;
                    }
                    block.type = 3;
                    this.obtainSolidBlock(block);
                    break;
                }
                case 5: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got key request from ", block.senderNodeId), 1);
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    if (session != null) {
                        session.makeBlockDeliveredByType(3);
                        session.makeBlockDeliveredByType(4);
                        session.makeBlockDeliveredByType(5);
                        Binder unbossedPayload = (Binder)Boss.load(block.payload);
                        byte[] signedUnbossed = unbossedPayload.getBinaryOrThrow("data");
                        if (session.publicKey != null) {
                            if (session.publicKey.verify(signedUnbossed, unbossedPayload.getBinaryOrThrow("signature"), HashType.SHA512)) {
                                UDPAdapter.this.report(this.getLabel(), "successfully verified ");
                                List receivedData = (List)Boss.load(signedUnbossed);
                                byte[] senderNonce = ((Bytes)receivedData.get(0)).toArray();
                                byte[] receiverNonce = ((Bytes)receivedData.get(1)).toArray();
                                if (Arrays.equals(receiverNonce, session.localNonce)) {
                                    Session.access$1402(session, senderNonce);
                                    if (session.state == 4) {
                                        session.createSessionKey();
                                        UDPAdapter.this.sendSessionKey(session);
                                        boolean sessionIsValid = session.isValid();
                                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" check session ", sessionIsValid));
                                        break;
                                    }
                                    int sessionRemoteNodeId = session.remoteNodeId;
                                    int sessionState = session.state;
                                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("node sent handshake too, to ", sessionRemoteNodeId, " state: ", sessionState), 1);
                                    if (session.state == 5) {
                                        if (UDPAdapter.this.myNodeInfo.getNumber() >= session.remoteNodeId) break;
                                        session.createSessionKey();
                                        UDPAdapter.this.sendSessionKey(session);
                                        boolean sessionIsValid = session.isValid();
                                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" check session ", sessionIsValid));
                                        break;
                                    }
                                    this.downStateAndResend(session);
                                    break;
                                }
                                UDPAdapter.this.sendHello(session);
                                throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": got nonce is not valid (not equals with current). Got nonce: " + receiverNonce + " from node " + block.senderNodeId + " with sender nonce " + senderNonce);
                            }
                            UDPAdapter.this.sendHello(session);
                            throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": sign has not verified. Got data have signed with key not match with known public key.");
                        }
                        UDPAdapter.this.sendHello(session);
                        throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": public key for current session is broken or does not exist");
                    }
                    block.type = 3;
                    this.obtainSolidBlock(block);
                    break;
                }
                case 6: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got session from ", block.senderNodeId), 1);
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    if (session != null) {
                        session.makeBlockDeliveredByType(3);
                        session.makeBlockDeliveredByType(4);
                        session.makeBlockDeliveredByType(5);
                        session.makeBlockDeliveredByType(6);
                        Binder unbossedPayload = (Binder)Boss.load(block.payload);
                        byte[] signedUnbossed = unbossedPayload.getBinaryOrThrow("data");
                        if (session.publicKey != null) {
                            if (session.publicKey.verify(signedUnbossed, unbossedPayload.getBinaryOrThrow("signature"), HashType.SHA512)) {
                                UDPAdapter.this.report(this.getLabel(), " successfully verified ");
                                byte[] decryptedData = UDPAdapter.this.ownPrivateKey.decrypt(signedUnbossed);
                                List receivedData = (List)Boss.load(decryptedData);
                                byte[] sessionKey = ((Bytes)receivedData.get(0)).toArray();
                                byte[] receiverNonce = ((Bytes)receivedData.get(1)).toArray();
                                if (Arrays.equals(receiverNonce, session.localNonce)) {
                                    session.reconstructSessionKey(sessionKey);
                                    boolean sessionIsValid = session.isValid();
                                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" check session ", sessionIsValid));
                                    this.answerAckOrNack(session, block, this.receivedDatagram.getAddress(), this.receivedDatagram.getPort());
                                    UDPAdapter.this.sendWaitingBlocks(session);
                                    break;
                                }
                                UDPAdapter.this.sendHello(session);
                                throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": got nonce is not valid (not equals with current)");
                            }
                            UDPAdapter.this.sendHello(session);
                            throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": sign has not verified. Got data have signed with key not match with known public key.");
                        }
                        UDPAdapter.this.sendHello(session);
                        throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": public key for current session does not exist");
                    }
                    block.type = 3;
                    this.obtainSolidBlock(block);
                    break;
                }
                case 0: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got data from ", block.senderNodeId), 1);
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    try {
                        if (session != null && session.isValid().booleanValue() && (session.state == 2 || session.state == 6)) {
                            session.makeBlockDeliveredByType(3);
                            session.makeBlockDeliveredByType(4);
                            session.makeBlockDeliveredByType(5);
                            session.makeBlockDeliveredByType(6);
                            Binder unbossedPayload = (Binder)Boss.load(block.payload);
                            int sessionKeyHashCode = session.sessionKey.hashCode();
                            int sessionRemoteNodeId = session.remoteNodeId;
                            UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("sessionKey is ", sessionKeyHashCode, " for ", sessionRemoteNodeId));
                            byte[] decrypted = session.sessionKey.etaDecrypt(unbossedPayload.getBinaryOrThrow("data"));
                            byte[] crc32Remote = unbossedPayload.getBinaryOrThrow("crc32");
                            byte[] crc32Local = new Crc32().digest(decrypted);
                            if (Arrays.equals(crc32Remote, crc32Local)) {
                                UDPAdapter.this.report(this.getLabel(), "Crc32 id ok", 1);
                                if (UDPAdapter.this.receiver != null) {
                                    UDPAdapter.this.receiver.accept(decrypted);
                                }
                            } else {
                                int sessionKeyHashCodeError = session.sessionKey.hashCode();
                                int sessionRemoteNodeIdError = session.remoteNodeId;
                                UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("Crc32 Error, sessionKey is ", sessionKeyHashCodeError, " for ", sessionRemoteNodeIdError), 1);
                                UDPAdapter.this.sendNack(session, block.blockId);
                                throw new EncryptionError((Object)((Object)Errors.BAD_VALUE) + ": Crc32 Error, decrypted length " + decrypted.length + "\n sessionKey is " + session.sessionKey.hashCode() + "\n own sessionKey is " + UDPAdapter.this.sessionKey.hashCode() + "\n string: " + new String(decrypted) + "\n remote id: " + session.remoteNodeId + "\n reconstructered block id: " + block.blockId);
                            }
                        }
                        this.answerAckOrNack(session, block, this.receivedDatagram.getAddress(), this.receivedDatagram.getPort());
                        break;
                    }
                    catch (SymmetricKey.AuthenticationFailed e) {
                        int sessionKeyHashCodeError = session.sessionKey.hashCode();
                        int sessionRemoteNodeIdError = session.remoteNodeId;
                        UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("SymmetricKey.AuthenticationFailed, sessionKey is ", sessionKeyHashCodeError, " for ", sessionRemoteNodeIdError), 1);
                        UDPAdapter.this.sendNack(session, block.blockId);
                        throw e;
                    }
                }
                case 1: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got ack from ", block.senderNodeId), 1);
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    int ackBlockId = (Integer)Boss.load(block.payload);
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("ackBlockId is: ", ackBlockId));
                    if (session == null) break;
                    int sendingPacketsQueueSize = session.sendingPacketsQueue.size();
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("num packets was in queue: ", sendingPacketsQueueSize));
                    session.makeBlockDelivered(ackBlockId);
                    session.removeBlockFromWaitingQueue(ackBlockId);
                    session.incremetWaitIndexForPacketsFromSendingQueue();
                    int sendingPacketsQueueSize2 = session.sendingPacketsQueue.size();
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage(" num packets in queue: ", sendingPacketsQueueSize2));
                    UDPAdapter.this.checkUnsentPackets(session);
                    if (session.state != 6) break;
                    session.state = 2;
                    UDPAdapter.this.sendWaitingBlocks(session);
                    break;
                }
                case 2: {
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got nack from ", block.senderNodeId), 1);
                    int ackBlockId = (Integer)Boss.load(block.payload);
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("blockId: ", ackBlockId));
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    if (session == null || !session.isValid().booleanValue() || session.state != 2 && session.state != 6) break;
                    session.moveBlocksFromSendingToWaiting();
                    session.removeDataBlocksFromWaiting();
                    session.state = 1;
                    int sessionKeyHashCode = session.sessionKey.hashCode();
                    int sessionRemoteNodeId = session.remoteNodeId;
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("sessionKey was ", sessionKeyHashCode, " for ", sessionRemoteNodeId));
                    session.sessionKey = UDPAdapter.this.sessionKey;
                    int sessionKeyHashCode2 = session.sessionKey.hashCode();
                    int sessionRemoteNodeId2 = session.remoteNodeId;
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("sessionKey now ", sessionKeyHashCode2, " for ", sessionRemoteNodeId2));
                    UDPAdapter.this.sendHello(session);
                    break;
                }
                case 7: {
                    List ackList = (List)Boss.load(block.payload);
                    int ackBlockId = (Integer)ackList.get(0);
                    int ackPacketId = (Integer)ackList.get(1);
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("got packet_ack from ", block.senderNodeId, " for block id ", ackBlockId, " for packet id ", ackPacketId));
                    session = (Session)UDPAdapter.this.sessionsById.get(block.senderNodeId);
                    if (session == null) break;
                    int sendingPacketsQueueSize = session.sendingPacketsQueue.size();
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("num packets was in queue: ", sendingPacketsQueueSize));
                    session.removePacketFromSendingQueue(ackBlockId, ackPacketId);
                    session.incremetWaitIndexForPacketsFromSendingQueue();
                    int sendingPacketsQueueSize2 = session.sendingPacketsQueue.size();
                    UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("num packets in queue: ", sendingPacketsQueueSize2));
                    UDPAdapter.this.checkUnsentPackets(session);
                }
            }
        }

        public void moveWaitingBlockToObtained(Block block) {
            this.waitingBlocks.remove(block.blockId);
            Instant blockExpiresAt = Instant.now().plus(this.maxObtainedBlockAge);
            this.obtainedBlocks.put(block.blockId, blockExpiresAt);
            this.obtainedBlocksQueue.add(new BlockTime(block.blockId, blockExpiresAt));
        }

        public void answerAckOrNack(Session session, Block block, InetAddress address, int port) throws InterruptedException {
            if (session != null && session.isValid().booleanValue() && (session.state == 2 || session.state == 6)) {
                UDPAdapter.this.sendAck(session, block.blockId);
            } else {
                String sessionToString = session != null ? session.toString() : "null";
                UDPAdapter.this.report(this.getLabel(), () -> UDPAdapter.this.concatReportMessage("answerAckOrNack ", sessionToString), 1);
                this.obtainedBlocks.remove(block.blockId);
                if (session != null) {
                    if (session.state == 2 || session.state == 6) {
                        UDPAdapter.this.sendNack(session, block.blockId);
                        if (UDPAdapter.this.sessionsById.containsKey(session.remoteNodeId)) {
                            UDPAdapter.this.sessionsById.remove(session.remoteNodeId);
                        }
                    }
                } else {
                    session = UDPAdapter.this.getOrCreateSession(block.senderNodeId, address, port);
                    UDPAdapter.this.sendNack(session, block.blockId);
                }
            }
        }

        public void downStateAndResend(Session session) throws InterruptedException {
            switch (session.state) {
                case 3: {
                    UDPAdapter.this.sendHello(session);
                }
                case 4: {
                    UDPAdapter.this.sendHello(session);
                    break;
                }
                case 5: {
                    UDPAdapter.this.sendWelcome(session);
                    break;
                }
                case 6: {
                    UDPAdapter.this.sendKeyRequest(session);
                }
            }
        }

        private class BlockTime {
            Integer blockId;
            Instant expiresAt;

            public BlockTime(Integer blockId, Instant expiresAt) {
                this.blockId = blockId;
                this.expiresAt = expiresAt;
            }
        }
    }
}

